// Copyright 2022 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/system/unified/date_tray.h"

#include "ash/constants/ash_features.h"
#include "ash/system/status_area_widget.h"
#include "ash/system/status_area_widget_test_helper.h"
#include "ash/system/time/time_tray_item_view.h"
#include "ash/system/time/time_view.h"
#include "ash/system/unified/unified_system_tray_bubble.h"
#include "ash/test/ash_test_base.h"
#include "base/test/metrics/histogram_tester.h"
#include "base/test/scoped_feature_list.h"
#include "base/time/time.h"
#include "base/time/time_override.h"

namespace ash {

class DateTrayTest : public AshTestBase {
 public:
  DateTrayTest() = default;
  DateTrayTest(const DateTrayTest&) = delete;
  DateTrayTest& operator=(const DateTrayTest&) = delete;
  ~DateTrayTest() override = default;

  void SetUp() override {
    // Enable calendar view feature.
    scoped_feature_list_.InitWithFeatures({ash::features::kCalendarView}, {});

    // Set time override.
    base::subtle::ScopedTimeClockOverrides time_override(
        []() {
          base::Time date;
          bool result = base::Time::FromString("24 Aug 2021 10:00 GMT", &date);
          DCHECK(result);
          return date;
        },
        /*time_ticks_override=*/nullptr,
        /*thread_ticks_override=*/nullptr);

    AshTestBase::SetUp();
    widget_ = CreateFramelessTestWidget();
    widget_->SetFullscreen(true);
    date_tray_ = StatusAreaWidgetTestHelper::GetStatusAreaWidget()->date_tray();
    widget_->SetContentsView(date_tray_);
    date_tray_->SetVisiblePreferred(true);
    date_tray_->unified_system_tray_->SetVisiblePreferred(true);
  }

  void TearDown() override {
    widget_.reset();
    date_tray_ = nullptr;
    AshTestBase::TearDown();
  }

  DateTray* GetDateTray() { return date_tray_; }

  UnifiedSystemTray* GetUnifiedSystemTray() {
    return date_tray_->unified_system_tray_;
  }

  std::u16string GetTimeViewText() {
    return date_tray_->time_view_->time_view()
        ->horizontal_label_date_for_test()
        ->GetText();
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
  std::unique_ptr<views::Widget> widget_;
  // Owned by `widget_`.
  DateTray* date_tray_ = nullptr;
};

// Test the initial state.
TEST_F(DateTrayTest, InitialState) {
  // Show the mock time now Month and day.
  EXPECT_EQ(u"Aug 24", GetTimeViewText());

  // Initial state: not showing the calendar bubble.
  EXPECT_FALSE(GetUnifiedSystemTray()->IsBubbleShown());
  EXPECT_FALSE(GetUnifiedSystemTray()->IsShowingCalendarView());
}

// Tests clicking/tapping the DateTray shows/closes the calendar bubble.
TEST_F(DateTrayTest, ShowCalendarBubble) {
  base::HistogramTester histogram_tester;
  // Clicking on the `DateTray` -> show the calendar bubble.
  LeftClickOn(GetDateTray());
  base::RunLoop().RunUntilIdle();

  EXPECT_TRUE(GetUnifiedSystemTray()->IsBubbleShown());
  EXPECT_TRUE(GetUnifiedSystemTray()->IsShowingCalendarView());

  histogram_tester.ExpectTotalCount("Ash.Calendar.ShowSource.TimeView", 1);

  // Clicking on the `DateTray` again -> close the calendar bubble.
  LeftClickOn(GetDateTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_FALSE(GetUnifiedSystemTray()->IsShowingCalendarView());

  // Tapping on the `DateTray` again -> open the calendar bubble.
  GestureTapOn(GetDateTray());
  base::RunLoop().RunUntilIdle();

  EXPECT_TRUE(GetUnifiedSystemTray()->IsBubbleShown());
  EXPECT_TRUE(GetUnifiedSystemTray()->IsShowingCalendarView());

  histogram_tester.ExpectTotalCount("Ash.Calendar.ShowSource.TimeView", 2);

  // Tapping on the `DateTray` again -> close the calendar bubble.
  GestureTapOn(GetDateTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_FALSE(GetUnifiedSystemTray()->IsBubbleShown());
  EXPECT_FALSE(GetUnifiedSystemTray()->IsShowingCalendarView());
}

// Tests the behavior when clicking on different areas.
TEST_F(DateTrayTest, ClickingArea) {
  // Clicking on the `DateTray` -> show the calendar bubble.
  LeftClickOn(GetDateTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_TRUE(GetUnifiedSystemTray()->IsShowingCalendarView());

  // Clicking on the bubble area -> not close the calendar bubble.
  LeftClickOn(GetUnifiedSystemTray()->bubble()->GetBubbleView());
  base::RunLoop().RunUntilIdle();
  EXPECT_TRUE(GetUnifiedSystemTray()->IsShowingCalendarView());

  // Clicking on the `UnifiedSystemTray` -> close the calendar bubble.
  LeftClickOn(GetUnifiedSystemTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_FALSE(GetUnifiedSystemTray()->IsBubbleShown());
  EXPECT_FALSE(GetUnifiedSystemTray()->IsShowingCalendarView());

  // Clicking on the `DateTray` -> show the calendar bubble.
  LeftClickOn(GetDateTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_TRUE(GetUnifiedSystemTray()->IsShowingCalendarView());

  // Clicking on the gap between `DateTray` and `UnifiedSystemTray`-> close the
  // bubble.
  auto* event_generator = GetEventGenerator();
  int date_tray_right = GetDateTray()->GetBoundsInScreen().right();
  int unigied_tray_left = GetUnifiedSystemTray()->GetBoundsInScreen().x();
  event_generator->MoveMouseTo(
      gfx::Point((date_tray_right + unigied_tray_left) / 2,
                 GetDateTray()->GetBoundsInScreen().CenterPoint().y()));
  event_generator->ClickLeftButton();
  LeftClickOn(GetDateTray());
  base::RunLoop().RunUntilIdle();
  EXPECT_FALSE(GetUnifiedSystemTray()->IsBubbleShown());
}

}  // namespace ash
