// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_KEY_CREATOR_IMPL_H_
#define ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_KEY_CREATOR_IMPL_H_

#include <memory>
#include <string>
#include <utility>

#include "ash/services/device_sync/cryptauth_key.h"
#include "ash/services/device_sync/cryptauth_key_bundle.h"
#include "ash/services/device_sync/cryptauth_key_creator.h"
#include "base/callback.h"
#include "base/containers/flat_map.h"
#include "base/memory/weak_ptr.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace ash {

namespace multidevice {
class SecureMessageDelegate;
}

namespace device_sync {

// Implementation of CryptAuthKeyCreator.
class CryptAuthKeyCreatorImpl : public CryptAuthKeyCreator {
 public:
  class Factory {
   public:
    static std::unique_ptr<CryptAuthKeyCreator> Create();
    static void SetFactoryForTesting(Factory* test_factory);

   protected:
    virtual ~Factory();
    virtual std::unique_ptr<CryptAuthKeyCreator> CreateInstance() = 0;

   private:
    static Factory* test_factory_;
  };

  CryptAuthKeyCreatorImpl(const CryptAuthKeyCreatorImpl&) = delete;
  CryptAuthKeyCreatorImpl& operator=(const CryptAuthKeyCreatorImpl&) = delete;

  ~CryptAuthKeyCreatorImpl() override;

  // CryptAuthKeyCreator:
  void CreateKeys(const base::flat_map<CryptAuthKeyBundle::Name, CreateKeyData>&
                      keys_to_create,
                  const absl::optional<CryptAuthKey>& server_ephemeral_dh,
                  CreateKeysCallback create_keys_callback) override;

 private:
  CryptAuthKeyCreatorImpl();

  void OnClientDiffieHellmanGenerated(const CryptAuthKey& server_ephemeral_dh,
                                      const std::string& public_key,
                                      const std::string& private_key);
  void OnDiffieHellmanHandshakeSecretDerived(const std::string& symmetric_key);

  // The Diffie-Hellman handshake secret, derived from the ephemeral server and
  // client keys, is null if no symmetric keys need to be created or if there
  // was an error deriving the handshake secret.
  void StartKeyCreation(
      const absl::optional<CryptAuthKey>& dh_handshake_secret);

  void OnAsymmetricKeyPairGenerated(CryptAuthKeyBundle::Name bundle_name,
                                    const std::string& public_key,
                                    const std::string& private_key);
  void OnSymmetricKeyDerived(CryptAuthKeyBundle::Name bundle_name,
                             const std::string& symmetric_key);

  size_t num_keys_to_create_ = 0;
  base::flat_map<CryptAuthKeyBundle::Name, CreateKeyData> keys_to_create_;
  base::flat_map<CryptAuthKeyBundle::Name, absl::optional<CryptAuthKey>>
      new_keys_;
  absl::optional<CryptAuthKey> client_ephemeral_dh_;
  CreateKeysCallback create_keys_callback_;

  std::unique_ptr<multidevice::SecureMessageDelegate> secure_message_delegate_;
};

}  // namespace device_sync

}  // namespace ash

#endif  // ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_KEY_CREATOR_IMPL_H_
