// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_DEVICE_SYNCER_IMPL_H_
#define ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_DEVICE_SYNCER_IMPL_H_

#include <memory>
#include <ostream>
#include <string>
#include <vector>

#include "ash/services/device_sync/cryptauth_device_registry.h"
#include "ash/services/device_sync/cryptauth_device_sync_result.h"
#include "ash/services/device_sync/cryptauth_device_syncer.h"
#include "ash/services/device_sync/cryptauth_ecies_encryptor.h"
#include "ash/services/device_sync/cryptauth_feature_status_getter.h"
#include "ash/services/device_sync/cryptauth_group_private_key_sharer.h"
#include "ash/services/device_sync/cryptauth_key.h"
#include "ash/services/device_sync/cryptauth_key_bundle.h"
#include "ash/services/device_sync/cryptauth_metadata_syncer.h"
#include "ash/services/device_sync/network_request_error.h"
#include "ash/services/device_sync/proto/cryptauth_better_together_device_metadata.pb.h"
#include "ash/services/device_sync/proto/cryptauth_devicesync.pb.h"
#include "ash/services/device_sync/proto/cryptauth_directive.pb.h"
#include "base/memory/weak_ptr.h"
#include "base/time/time.h"
#include "base/timer/timer.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

class PrefService;

namespace cryptauthv2 {
class ClientAppMetadata;
class ClientMetadata;
}  // namespace cryptauthv2

namespace ash {

namespace device_sync {

class AttestationCertificatesSyncer;
class CryptAuthClient;
class CryptAuthClientFactory;
class CryptAuthKeyRegistry;
class SyncedBluetoothAddressTracker;

// An implementation of CryptAuthDeviceSyncer, using instances of
// CryptAuthClient to make the API calls to CryptAuth. This implementation
// handles timeouts internally, so the callback passed to
// CryptAuthDeviceSyncer::Sync() is always guaranteed to be invoked.
//
// When the DeviceSync flow finishes, the device registry is updated with all
// devices that have a valid device ID, device name, device public key, and
// feature states. If device metadata cannot be decrypted due to an error or
// because the group private key was not returned by CryptAuth, the device is
// still added to the registry without the new decrypted metadata. Any existing
// decrypted metadata from the device registry will remain there until the new
// metadata can be decrypted.
class CryptAuthDeviceSyncerImpl : public CryptAuthDeviceSyncer {
 public:
  class Factory {
   public:
    static std::unique_ptr<CryptAuthDeviceSyncer> Create(
        CryptAuthDeviceRegistry* device_registry,
        CryptAuthKeyRegistry* key_registry,
        CryptAuthClientFactory* client_factory,
        SyncedBluetoothAddressTracker* synced_bluetooth_address_tracker,
        AttestationCertificatesSyncer* attestation_certificates_syncer,
        PrefService* pref_service,
        std::unique_ptr<base::OneShotTimer> timer =
            std::make_unique<base::OneShotTimer>());
    static void SetFactoryForTesting(Factory* test_factory);

   protected:
    virtual ~Factory();
    virtual std::unique_ptr<CryptAuthDeviceSyncer> CreateInstance(
        CryptAuthDeviceRegistry* device_registry,
        CryptAuthKeyRegistry* key_registry,
        CryptAuthClientFactory* client_factory,
        SyncedBluetoothAddressTracker* synced_bluetooth_address_tracker,
        AttestationCertificatesSyncer* attestation_certificates_syncer,
        PrefService* pref_service,
        std::unique_ptr<base::OneShotTimer> timer) = 0;

   private:
    static Factory* test_factory_;
  };

  CryptAuthDeviceSyncerImpl(const CryptAuthDeviceSyncerImpl&) = delete;
  CryptAuthDeviceSyncerImpl& operator=(const CryptAuthDeviceSyncerImpl&) =
      delete;

  ~CryptAuthDeviceSyncerImpl() override;

 private:
  enum class State {
    kNotStarted,
    kWaitingForBluetoothAddress,
    kWaitingForAttestationCertificates,
    kWaitingForMetadataSync,
    kWaitingForFeatureStatuses,
    kWaitingForEncryptedGroupPrivateKeyProcessing,
    kWaitingForEncryptedDeviceMetadataProcessing,
    kWaitingForGroupPrivateKeySharing,
    kFinished
  };

  friend std::ostream& operator<<(std::ostream& stream, const State& state);

  static absl::optional<base::TimeDelta> GetTimeoutForState(State state);
  static absl::optional<CryptAuthDeviceSyncResult::ResultCode>
  ResultCodeErrorFromTimeoutDuringState(State state);

  // |device_registry|: At the end of a DeviceSync flow, the devices in the
  //     registry are replaced with the devices received from CryptAuth.
  // |key_registry|: The syncer will read and possibly write the group key pair,
  //     and it will read the user key pair and the key used for decrypting the
  //     group private key.
  // |client_factory|: Creates CryptAuthClient instances for making API calls.
  // |synced_bluetooth_address_tracker|: Used to fetch Bluetooth address and
  //     track address used for successful syncs.
  // |timer|: Handles timeouts for asynchronous operations.
  CryptAuthDeviceSyncerImpl(
      CryptAuthDeviceRegistry* device_registry,
      CryptAuthKeyRegistry* key_registry,
      CryptAuthClientFactory* client_factory,
      SyncedBluetoothAddressTracker* synced_bluetooth_address_tracker,
      AttestationCertificatesSyncer* attestation_certificates_syncer,
      PrefService* pref_service,
      std::unique_ptr<base::OneShotTimer> timer);

  // CryptAuthDeviceSyncer:
  void OnAttemptStarted(
      const cryptauthv2::ClientMetadata& client_metadata,
      const cryptauthv2::ClientAppMetadata& client_app_metadata) override;

  void SetState(State state);
  void OnTimeout();

  // Controls the logical flow of the class.
  void AttemptNextStep();

  void GetBluetoothAddress();
  void OnBluetoothAddress(const std::string& bluetooth_address);

  bool IsAttestationCertificatesUpdateRequired();
  void GetAttestationCertificates();
  void OnAttestationCertificates(const std::vector<std::string>& cert_chain);

  void SyncMetadata();
  void OnSyncMetadataFinished(
      const CryptAuthMetadataSyncer::IdToDeviceMetadataPacketMap&
          id_to_device_metadata_packet_map,
      std::unique_ptr<CryptAuthKey> new_group_key,
      const absl::optional<cryptauthv2::EncryptedGroupPrivateKey>&
          encrypted_group_private_key,
      const absl::optional<cryptauthv2::ClientDirective>& new_client_directive,
      CryptAuthDeviceSyncResult::ResultCode device_sync_result_code);

  void SetGroupKey(const CryptAuthKey& new_group_key);

  void GetFeatureStatuses();
  void OnGetFeatureStatusesFinished(
      const CryptAuthFeatureStatusGetter::IdToDeviceSoftwareFeatureInfoMap&
          id_to_device_software_feature_info_map,
      CryptAuthDeviceSyncResult::ResultCode device_sync_result_code);

  // Builds a new device registry map with all device information except
  // decrypted BetterTogetherDeviceMetadata for remote devices.
  void BuildNewDeviceRegistry(
      const CryptAuthFeatureStatusGetter::IdToDeviceSoftwareFeatureInfoMap&
          id_to_device_software_feature_info_map);

  // If an encrypted group private key was sent by CryptAuth, decrypt it. Even
  // if we already have the unencrypted group private key in the key registry,
  // we verify that they agree.
  void ProcessEncryptedGroupPrivateKey();
  void OnGroupPrivateKeyDecrypted(
      const absl::optional<std::string>& group_private_key_from_cryptauth);

  void ProcessEncryptedDeviceMetadata();
  void OnDeviceMetadataDecrypted(const CryptAuthEciesEncryptor::IdToOutputMap&
                                     id_to_decrypted_metadata_map);

  // Adds decrypted BetterTogetherDeviceMetadata to the new device registry
  // constructed in BuildNewDeviceRegistry().
  void AddDecryptedMetadataToNewDeviceRegistry(
      const CryptAuthEciesEncryptor::IdToOutputMap&
          id_to_decrypted_metadata_map);

  void ShareGroupPrivateKey();
  void OnShareGroupPrivateKeyFinished(
      CryptAuthDeviceSyncResult::ResultCode device_sync_result_code);

  // Replaces the current device registry if devices were able to be extracted
  // from the DeviceSync attempt. Finishes the DeviceSync attempt, sending back
  // the relevant CryptAuthDeviceSyncResult.
  void FinishAttempt(CryptAuthDeviceSyncResult::ResultCode result_code);

  bool did_non_fatal_error_occur_ = false;

  // Set in OnAttemptStarted() and not modified during the rest of the flow.
  cryptauthv2::RequestContext request_context_;
  cryptauthv2::BetterTogetherDeviceMetadata
      local_better_together_device_metadata_;

  // Output from CryptAuthMetadataSyncer.
  CryptAuthMetadataSyncer::IdToDeviceMetadataPacketMap
      id_to_device_metadata_packet_map_;
  absl::optional<cryptauthv2::EncryptedGroupPrivateKey>
      encrypted_group_private_key_;
  absl::optional<cryptauthv2::ClientDirective> new_client_directive_;

  // Populated after a successful BatchGetFeatureStatuses call. Device metadata
  // is added if device metadata decryption is successful. Replaces the contents
  // of the device registry if non-null when the DeviceSync attempt ends,
  // successfully or not.
  absl::optional<CryptAuthDeviceRegistry::InstanceIdToDeviceMap>
      new_device_registry_map_;

  // The time of the last state change. Used for execution time metrics.
  base::TimeTicks last_state_change_timestamp_;

  std::unique_ptr<CryptAuthMetadataSyncer> metadata_syncer_;
  std::unique_ptr<CryptAuthFeatureStatusGetter> feature_status_getter_;
  std::unique_ptr<CryptAuthEciesEncryptor> encryptor_;
  std::unique_ptr<CryptAuthGroupPrivateKeySharer> group_private_key_sharer_;

  State state_ = State::kNotStarted;
  CryptAuthDeviceRegistry* device_registry_ = nullptr;
  CryptAuthKeyRegistry* key_registry_ = nullptr;
  CryptAuthClientFactory* client_factory_ = nullptr;
  SyncedBluetoothAddressTracker* synced_bluetooth_address_tracker_ = nullptr;
  AttestationCertificatesSyncer* attestation_certificates_syncer_ = nullptr;
  PrefService* pref_service_ = nullptr;
  std::unique_ptr<base::OneShotTimer> timer_;

  base::WeakPtrFactory<CryptAuthDeviceSyncerImpl> weak_ptr_factory_{this};
};

}  // namespace device_sync

}  // namespace ash

#endif  // ASH_SERVICES_DEVICE_SYNC_CRYPTAUTH_DEVICE_SYNCER_IMPL_H_
