// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_COMPONENTS_LOGIN_AUTH_EXTENDED_AUTHENTICATOR_IMPL_H_
#define ASH_COMPONENTS_LOGIN_AUTH_EXTENDED_AUTHENTICATOR_IMPL_H_

#include <memory>
#include <string>

#include "ash/components/login/auth/extended_authenticator.h"
#include "base/callback.h"
#include "base/compiler_specific.h"
#include "base/component_export.h"
#include "chromeos/dbus/cryptohome/UserDataAuth.pb.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "third_party/cros_system_api/dbus/service_constants.h"

namespace ash {

class AuthStatusConsumer;
class UserContext;

// Implements ExtendedAuthenticator.
class COMPONENT_EXPORT(ASH_LOGIN_AUTH) ExtendedAuthenticatorImpl
    : public ExtendedAuthenticator {
 public:
  static scoped_refptr<ExtendedAuthenticatorImpl> Create(
      AuthStatusConsumer* consumer);

  ExtendedAuthenticatorImpl(const ExtendedAuthenticatorImpl&) = delete;
  ExtendedAuthenticatorImpl& operator=(const ExtendedAuthenticatorImpl&) =
      delete;

  // ExtendedAuthenticator:
  void SetConsumer(AuthStatusConsumer* consumer) override;
  void AuthenticateToCheck(const UserContext& context,
                           base::OnceClosure success_callback) override;
  void AuthenticateToUnlockWebAuthnSecret(
      const UserContext& context,
      base::OnceClosure success_callback) override;
  void StartFingerprintAuthSession(
      const AccountId& account_id,
      base::OnceCallback<void(bool)> callback) override;
  void EndFingerprintAuthSession() override;
  void AuthenticateWithFingerprint(
      const UserContext& context,
      base::OnceCallback<void(user_data_auth::CryptohomeErrorCode)> callback)
      override;
  void TransformKeyIfNeeded(const UserContext& user_context,
                            ContextCallback callback) override;

 private:
  explicit ExtendedAuthenticatorImpl(AuthStatusConsumer* consumer);
  ~ExtendedAuthenticatorImpl() override;

  // Callback for system salt getter.
  void OnSaltObtained(const std::string& system_salt);

  // Performs actual operation with fully configured |context|.
  void DoAuthenticateToCheck(base::OnceClosure success_callback,
                             bool unlock_webauthn_secret,
                             const UserContext& context);

  // Inner operation callbacks.
  template <typename ReplyType>
  void OnOperationComplete(const char* time_marker,
                           const UserContext& context,
                           base::OnceClosure success_callback,
                           absl::optional<ReplyType> reply);

  void OnStartFingerprintAuthSessionComplete(
      base::OnceCallback<void(bool)> callback,
      absl::optional<user_data_auth::StartFingerprintAuthSessionReply> reply);
  void OnFingerprintScanComplete(
      base::OnceCallback<void(user_data_auth::CryptohomeErrorCode)> callback,
      absl::optional<user_data_auth::CheckKeyReply> reply);

  bool salt_obtained_;
  std::string system_salt_;
  std::vector<base::OnceClosure> system_salt_callbacks_;

  AuthStatusConsumer* consumer_;
};

}  // namespace ash

#endif  // ASH_COMPONENTS_LOGIN_AUTH_EXTENDED_AUTHENTICATOR_IMPL_H_
